/***********************************************************
 * http-server.cpp                                         *
 *                                                         *
 * Pier Guillen (pguillen)                                 *
 * CS 536 - Assignment 1                                   *
 * Feb 7, 2011                                             *
 ***********************************************************/

#include "http-server.h"
#include "logger.h"
#include <arpa/inet.h>
#include <cerrno>
#include <cstring>
using namespace std;


// Initializes the server
void HttpServer::init(string mode, string port) {
   // Initialize fields
   httpMode = mode;
   if ((httpMode != "1.0") && (httpMode != "1.1")) {
      httpMode = Const::DEFAULT_MODE;
   }
   // Initialize socket to the given port
   addrinfo *servInfo;
   servInfo = localAddressInfo(port);
   findBind(servInfo);
   freeaddrinfo(servInfo);
   startListening();
   Logger::log.write(Logger::INFO, "Server initialized on port " + port);
   Logger::log.write(Logger::INFO, "Server initialized on mode " + httpMode);
}


// Starts the server
void HttpServer::start() {
   while (true) {
//cout << "Start loop" << endl;
      // Listen for new connections
      fd_set readUsers = listenForConnections();
      acceptConnections(&readUsers);
      // Give service to connect clients
      for (unsigned int i = 0; i < clients.size(); i++) {
         clients[i]->listenForRequests(&readUsers);
         clients[i]->receiveRequest();
         clients[i]->parseRequest();
         clients[i]->appendRequest();
         clients[i]->processRequest();
         clients[i]->transmitContent();
         int timeout = Const::CONNECTION_TIMEOUT/clients.size() + 1;
         if (clients[i]->closeConnection(&connectedUsers, timeout)) {
            delete clients[i];
            clients.erase(clients.begin() + i);
         }
      }
      Logger::log.flush();
//cout << "End loop" << endl;
   }
}


// Gets information about host address
addrinfo* HttpServer::localAddressInfo(string port) {
   int status;
   addrinfo hints;
   addrinfo* servInfo;
   memset(&hints, 0, sizeof hints);
   hints.ai_family = AF_UNSPEC;           // Use either IPv4 or IPv6
   hints.ai_socktype = SOCK_STREAM;       // Stream sockets (required for TCP)
   hints.ai_flags = AI_PASSIVE;           // Indicates we'll use the address to bind
   if ((status = getaddrinfo(NULL, port.c_str(), &hints, &servInfo)) != 0) {
      string error = "Error while getting host address information : ";
      error += gai_strerror(status);
      throw error;
   }
   return servInfo;
}


// Gets whether the socket address is IPv4 or IPv6
inline void* HttpServer::getInAddr(sockaddr *socketAddr) {
   if (socketAddr->sa_family == AF_INET) {
      return &(((sockaddr_in*)socketAddr)->sin_addr);
   }
   return &(((sockaddr_in6*)socketAddr)->sin6_addr);
}


// Searches through the list of host infomation for the first it can bind to
void HttpServer::findBind(addrinfo *servInfo) {
   bool binded = false;
   for (addrinfo *currInfo = servInfo; currInfo != NULL; currInfo = currInfo->ai_next) {
      // Get a socket description for the current info
      socketDescriptor = socket(currInfo->ai_family, currInfo->ai_socktype, currInfo->ai_protocol);
      if (socketDescriptor == -1) {
         continue;
      }
      // Set SO_REUSEADDR on the socket to true (1), which is needed to bind
      int optVal = 1;
      if (setsockopt(socketDescriptor, SOL_SOCKET, SO_REUSEADDR, &optVal, sizeof(int)) == -1) {
         throw string("Unable to set socket options.");
      }
      // Bind the socket to IP address and port number
      if (bind(socketDescriptor, currInfo->ai_addr, currInfo->ai_addrlen) == -1) {
         close(socketDescriptor);
         continue;
      }
      binded = true;
      break;
   }
   // Check if the socket was succesfully binded
   if (!binded) {
      throw string("Failed to bind socket.");
   }
}


// Starts listening for incoming connections
void HttpServer::startListening() {
   // Start listening
   if (listen(socketDescriptor, Const::QUEUE_SIZE) == -1) {
      throw string("Error : unable to start listening for incoming connections.");
   }
   // Clear the connection set and add the current descriptor
   FD_ZERO(&connectedUsers);
   FD_SET(socketDescriptor, &connectedUsers);
   maxDescriptor = socketDescriptor;
}


// Listen the ports for connections
fd_set HttpServer::listenForConnections() {
   timeval timeout;
   timeout.tv_sec = 0;
   timeout.tv_usec = Const::LISTEN_TIMEOUT;
   fd_set readUsers = connectedUsers;
   if (select(maxDescriptor + 1, &readUsers, NULL, NULL, &timeout) == -1) {
      string error = "Error when polling for connections.\n";
      error+= strerror(errno);
      Logger::log.write(Logger::ERROR, error);
   }
   return readUsers;
}


// Accept new connections 
void HttpServer::acceptConnections(fd_set* readUsers) {
   // If we have a new read request in our connection port, check for new connections
   if (FD_ISSET(socketDescriptor, readUsers)) {
      sockaddr_storage remoteAddr;
      socklen_t addrLen = sizeof remoteAddr;
      int newDescriptor = accept(socketDescriptor, (sockaddr *) &remoteAddr, &addrLen);
      // Add new descriptor to set and keep track of the max
      if (newDescriptor != -1) {
          Logger::log.write(Logger::INFO, "Accepted connection", newDescriptor);
         // Get client address
         char address[INET6_ADDRSTRLEN];
         inet_ntop(remoteAddr.ss_family, getInAddr((sockaddr *) &remoteAddr), address, sizeof address);
         // Create new connection
         clients.push_back(new Client(newDescriptor, address, httpMode));
         FD_SET(newDescriptor, readUsers);
         FD_SET(newDescriptor, &connectedUsers);
         if (newDescriptor > maxDescriptor) {
            maxDescriptor = newDescriptor;
         }
      } else {
         string error = "Error when accepting new connection.\n";
         error+= strerror(errno);
         Logger::log.write(Logger::ERROR, error);
      }
   }
}
