/***********************************************************
 * client.cpp                                              *
 *                                                         *
 * Pier Guillen (pguillen)                                 *
 * CS 536 - Assignment 1                                   *
 * Feb 7, 2011                                             *
 ***********************************************************/

#include "client.h"
#include "constants.h"
#include "logger.h"
#include <sys/stat.h>
#include <algorithm>
#include <cerrno> 
#include <cstring>
#include <sstream> 
using namespace std;


// Constructor
Client::Client(int socketDescriptor, string clientAddr, string httpMode) {
   path = "";
   method = "";
   version = "";
   response = "";
   entityBody = "";
   currState = LISTEN;
   address = clientAddr;
   descriptor = socketDescriptor;
   serverHttpMode = "HTTP/" + httpMode;
   Logger::log.write(Logger::INFO, "New connection from: " + address + " on port ", descriptor);
}


// Listen for incoming requests
void Client::listenForRequests(fd_set* readUsers) {
   if ((currState == LISTEN) && (FD_ISSET(descriptor, readUsers))) {
      currState = RECEIVE;
      Logger::log.write(Logger::INFO, "Listen:", descriptor);
   }
}


// Receive incoming request from the client
void Client::receiveRequest() {
   if (currState != RECEIVE) {
      return;
   }
   // Read and store the request
   Logger::log.write(Logger::INFO, "Receive:", descriptor);
   char* buffer = new char[Const::BUFFER_SIZE];
   ssize_t byteCount = recv(descriptor, buffer, Const::BUFFER_SIZE, 0);
   if (byteCount > 0) {
      request = buffer;
      currState = PARSE;
   } else if (byteCount == 0) {
      currState = CLOSE;
      Logger::log.write(Logger::INFO, "Client closed connection.");
   } else {
      response = createErrorResponse(505);
      currState = TRANSMIT;
      string error = "Error while receiving request.\n";
      error+= strerror(errno);
      Logger::log.write(Logger::ERROR, error);
   }
   delete[] buffer;
}


// Parse the client request
void Client::parseRequest() {
   if (currState != PARSE) {
      return;
   }
   Logger::log.write(Logger::INFO, "Parse:", descriptor);
   // Parse request-line
   istringstream inStream(request);
   inStream >> method >> path >> version >> ws;
   Logger::log.write(Logger::INFO, 
      "Path: " + path + "\n" + 
      "Method: " + method + "\n" +
      "Version: " + version);
   // Obtain headers      
   connectionHeader = "";
   contentLenHeader = "0";
   contentTypeHeader = "";
   dateHeader = "";
   hostHeader = "";
   string header;
   while (getline(inStream, header)) {
      if (header.find("Content-Length:") == 0) {
         contentLenHeader = header;
         Logger::log.write(Logger::INFO, header);
      } else if (header.find("Content-Type:") == 0) {
         contentTypeHeader = header;
         Logger::log.write(Logger::INFO, header);
      } else if (header.find("Date:") == 0) {
         dateHeader = header;
         Logger::log.write(Logger::INFO, header);
      } else if (header == "") {
         break;
      } else if (serverHttpMode.find("1.1")) {
         // Headers exclusive of HTTP 1.1
         if (header.find("Connection:") == 0) {
            connectionHeader = header;
            Logger::log.write(Logger::INFO, header);
         } else if (header.find("Host:") == 0) {
            hostHeader = header;
            Logger::log.write(Logger::INFO, header);
         }
      }
   }
   // Get the body of the request
   inStream >> entityBody;
   // Parse the requested path
   if (path[path.length() -1] == '/') {
      path+= Const::DEFAULT_FILE;
   }
   size_t pos = path.find('\\');
   while (pos != string::npos) {
      path[pos] = '/';
      pos = path.find('\\');
   }
   if (path[0] == '/') {
      path.erase(0, 1);
   }
   path = Const::ROOT + path;
   Logger::log.write(Logger::INFO, "Path modified to: " + path);
   // Append anything we might have missed on the first receive
   currState = APPEND;
}


// Append information to the body of the request
void Client::appendRequest() {
   if (currState != APPEND) {
      return;
   }
   Logger::log.write(Logger::INFO, "Append:", descriptor);
   // Get the size of the body
   size_t contentLength = 0;
   size_t pos = contentLenHeader.find(":");
   if (pos != string::npos) {
      contentLength = atoi(contentLenHeader.substr(pos).c_str());
   }
   // Check if we have read the body completely
   if (contentLength > entityBody.size()) {
      // Still not complete
      char* buffer = new char[Const::BUFFER_SIZE];
      ssize_t byteCount = recv(descriptor, buffer, Const::BUFFER_SIZE, 0);
      if (byteCount > 0) {
         entityBody+= buffer;
      } else if (byteCount == 0) {
         currState = CLOSE;
         Logger::log.write(Logger::INFO, "Client closed connection.");
      } else {
         response = createErrorResponse(505);
         currState = TRANSMIT;
         string error = "Error while receiving request.\n";
         error+= strerror(errno);
         Logger::log.write(Logger::ERROR, error);
      }
      delete[] buffer;
   } else {
      // Process the request
      Logger::log.write(Logger::INFO, "Entity body size:", entityBody.size());
      currState = PROCESS;
   }
}


// Process the client's request
void Client::processRequest() {
   if (currState != PROCESS) {
      return;
   }
   Logger::log.write(Logger::INFO, "Process:", descriptor);
   response = "";
   // Check HTTP version
   if ((serverHttpMode.find("1.0") != string::npos) && 
       (serverHttpMode.find(version) == string::npos)) {
      response = createErrorResponse(505);
      currState = TRANSMIT;
      return;
   }
   if ((version.find("1.1") != string::npos) && (hostHeader == "")) {
      response = createErrorResponse(400);
      currState = TRANSMIT;
      return;
   }
   // Respond according to request type
   if (method == "GET") {
      processGet();
   } else if (method  == "HEAD") {
      processHead();
   } else if (method  == "PUT") {
      processPut();
   } else if (method  == "DELETE") {
      processDelete();
   } else if (method  == "TRACE") {
      processTrace();
   } else if (method  == "OPTIONS") {
      processOptions();
   } else {
      response = createErrorResponse(400);
   }
   // Transmit the response
   currState = TRANSMIT;
}


// Transmit content to the client
void Client::transmitContent() {
   if (currState != TRANSMIT) {
      return;
   }
   // Send information
   Logger::log.write(Logger::INFO, "Transmit:", descriptor);
   size_t len = (response.size() < Const::BUFFER_SIZE) ? (response.size()) : (Const::BUFFER_SIZE);
   ssize_t byteCount = send(descriptor, response.c_str(), len, 0);
   if (byteCount == -1) {
      string error = "Error while sending information.\n";
      error+= strerror(errno);
      Logger::log.write(Logger::ERROR, error);
   }
   // Check if we're done
   response.erase(0, byteCount);
   if (response.size() == 0) {
      startTime = time(NULL);
      currState = CLOSE;
   }
}


// Close unused connections from clients
bool Client::closeConnection(fd_set* connectedUsers, int disconnectTime) {
   // Don't close if there are pending operations
   if ((currState != LISTEN) && (currState != CLOSE)) {
      return false;
   }
   // If using HTTP 1.1 and not enough time has passed, keep connection
   time_t elapsedTime = time(NULL) -startTime;
   if ((serverHttpMode.find("1.1") != string::npos) && 
       (connectionHeader.find("keep-alive") != string::npos)) {
      if (elapsedTime < disconnectTime) {
         currState = LISTEN;
         Logger::log.write(Logger::INFO, "Keeping connection alive:", descriptor);
         return false;
      } else {
         Logger::log.write(Logger::INFO, "Connection kept alive for (sec):", elapsedTime);
      }
   }
   // Close connection with client
   Logger::log.write(Logger::INFO, "Close:", descriptor);
   if (close(descriptor) != -1) {
      FD_CLR(descriptor, connectedUsers);
      return true;
   } else {
      Logger::log.write(Logger::ERROR, "Error while closing");
      return false;
   }
}


// Creates the response to a successful request
string Client::createResponse(string resource, string type) {
   size_t  size = resource.length();
   stringstream response;
   response << serverHttpMode << Const::SP << Const::CODE_200 << Const::CRLF
      << "Content-Type: " << type << Const::CRLF
      << "Content-Length: " << size << Const::CRLF
      << Const::CRLF
      << resource;
   return response.str();
}


// Creates the response to be sent when encountering an error
string Client::createErrorResponse(int error) {
   string html = createErrorHtml(error);
   int size = html.length();
   stringstream response;
   response << serverHttpMode << Const::SP << getError(error) << Const::CRLF
      << "Content-Type: " << Const::TYPE_HTML << Const::CRLF
      << "Content-Length: " << size << Const::CRLF
      << Const::CRLF
      << html;
   return response.str();
}


// Gets a string with status code and reason phrase of the error
string Client::getError(int error) {
   switch (error) {
      case 400: 
         return Const::CODE_400;
      case 403:
         return Const::CODE_403;
      case 404:
         return Const::CODE_404;
      case 505:
         return Const::CODE_505;         
   }
   return Const::CODE_500;
}


// Creates an html file containg the provided error
string Client::createErrorHtml(int error) {
   string result;
   result = "<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n";
   result+= "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n";
   result+= "      \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">\n";
   result+= "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n";
   result+= "<head>\n";
   result+= "  <meta http-equiv=\"content-type\" content=\"text/html; charset=iso-8859-1\" />\n";
   result+= "  <title>Error</title>\n";
   result+= "</head>\n";
   result+= "\n";
   result+= "<body>\n";
   result+= "<p>" + getError(error) + "</p>\n";
   result+= "</body>\n";
   result+= "</html>\n";

   return result;
}


// Process a GET/HEAD request
void Client::processGetHead(bool justHead) {
   // Check if the requested file exist
   struct stat fileInfo;
   int intStat = stat(path.c_str(), &fileInfo);
   if (intStat == 0) {
      // Check if we have permission for that file
      if ((fileInfo.st_mode & S_IRUSR) == 0) {
         response = createErrorResponse(403); 
         Logger::log.write(Logger::WARNING, "Unable to read file (no permissions) " + path);
         return;
      }
      // Open file
      ifstream inFile;
      inFile.open(path.c_str());
      if (inFile) {
         string type = Const::TYPE_HTML;
         if (path.find(".jpg") != string::npos){
            type = Const::TYPE_JPG;
         }
         if (path.find(".gif") != string::npos){
            type = Const::TYPE_GIF;
         }
         // Just need the head
         if (justHead) {
            response = createResponse("", type);
            return;
         }
         // It exists, read it and store it
         ostringstream out;
         out << inFile.rdbuf();
         response = createResponse(out.str(), type);
         inFile.close();
      } else {
         // File exists but cannot open it
         response = createErrorResponse(500); 
         Logger::log.write(Logger::WARNING, "Unable to open file " + path);
      }
   } else {
      // Files doesn't exist
      response = createErrorResponse(404); 
      Logger::log.write(Logger::WARNING, "Unable to find file " + path);
   }
}


// Process a GET request
void Client::processGet() {
   processGetHead(false);
}


// Process a HEAD request
void Client::processHead() {
   processGetHead(true);
}


// Process a PUT request
void Client::processPut() {
   // Check if we have permission for that file
   struct stat fileInfo;
   int intStat = stat(path.c_str(), &fileInfo);
   if ((intStat == 0) && ((fileInfo.st_mode & S_IWUSR) == 0)) {
      response = createErrorResponse(403); 
      Logger::log.write(Logger::WARNING, "Unable to write file (no permissions) " + path);
      return;
   }
   // Open file
   ofstream outFile;
   outFile.open(path.c_str());
   outFile << entityBody;
   outFile.close();
}


// Process a DELETE request
void Client::processDelete() {
   // Check if the requested file exist
   struct stat fileInfo;
   int intStat = stat(path.c_str(), &fileInfo);
   if (intStat == 0) {
      // Check if we have permission for that file
      if ((fileInfo.st_mode & S_IWUSR) == 0) {
         response = createErrorResponse(403); 
         Logger::log.write(Logger::WARNING, "Unable to delete file (no permissions) " + path);
         return;
      }
      // Delete file
      if (remove(path.c_str()) == 0) {
         response = version + Const::SP + Const::CODE_204 + Const::CRLF + Const::CRLF;
         Logger::log.write(Logger::INFO, "File deleted " + path);
      } else  {
         // File exists but cannot delete it
         response = createErrorResponse(500); 
         Logger::log.write(Logger::WARNING, "Unable to delete file " + path);
         Logger::log.write(Logger::WARNING, strerror(errno));
      }
   } else {
      // Files doesn't exist
      response = createErrorResponse(404); 
      Logger::log.write(Logger::WARNING, "Unable to find file " + path);
   }
}


// Process a TRACE request
void Client::processTrace() {
   response = createResponse(request, Const::TYPE_MESSAGE);
}


// Process an OPTION request
void Client::processOptions() {
   int size = 0;
   string allowed;
   stringstream stream;
   if (path == "*") {
      allowed = "GET HEAD PUT DELETE TRACE OPTIONS";
   } else {
      allowed = "TRACE OPTIONS";
      // Check if the requested file exist
      struct stat fileInfo;
      int intStat = stat(path.c_str(), &fileInfo);
      if (intStat == 0) {
         // Check if we have permission for writing
         if ((fileInfo.st_mode & S_IWUSR) != 0) {
            allowed = "PUT DELETE " + allowed;
         }
         // Check if we have permission for reading
         if ((fileInfo.st_mode & S_IRUSR) != 0) {
            allowed = "GET HEAD" + allowed;
         }
      } else {
         // Files doesn't exist
         response = createErrorResponse(404); 
         Logger::log.write(Logger::WARNING, "Unable to find file " + path);
      }
   }
   // Create response
   stream << version << Const::SP << Const::CODE_200 << Const::CRLF
      << "Allow: " << allowed << Const::CRLF
      << "Content-Type: " << Const::TYPE_HTML << Const::CRLF
      << "Content-Length:" << size << Const::CRLF
      << Const::CRLF;
   response = stream.str();
}
